// 	===============================================================
// 	ESTRUC.CPY - FC7 File Format Entity Definitions
// 	===============================================================
// 	Copyright 1999 Evolution Computing
// 	All rights reserved
// 	Written by Mike Riddle 21-July-1999
// 	===============================================================

#ifndef _ESTRUC_H_
#define _ESTRUC_H_

#pragma pack(1)

#include <DStyles.h>
#include <TForm.h>

// 	Entity Class flags returned by EInfo EI_Class

#define EIC_Point			1	// 1=point - can not be scanned
#define EIC_Linear		2	// 1=can be scanned T=0 to T=1
#define EIC_Area			4	// 1=entity encloses area
#define EIC_Volume		8	// 1=entity encloses volume
#define EIC_3d			 16	// 0=2d, 1=3d
#define EIC_Multi		 32	// 1=multiple-entity aggregate
#define EIC_Note		 64	// 1=annotation entry (not geometry)
#define EIC_Curve		128	// 1=ent is curve that should be smoothed

// 	Entity Handler Caps flags returned by EInfo EI_Caps

#define EIC_Std				0	// only does rired functions
#define EIC_NoDraw		1	// 1=entity normally does not draw
#define EIC_NUTGrow		2	// entity can grow on non-uniform tran

// 	===============================================================
// 	Common Stuff - Common to all drawable entities (32 bytes)
// 	===============================================================

typedef struct
{
	DWORD	ERLen;
	char	EType;		
	char	EThick;			// pen thickness 0->2.54 mm paper scale
	DWORD	EColor;			// outline color
	DWORD	EColor2;		// fill color
	WORD	EFStyle;		// 0 hollow, 1 solid
	WORD	ELStyle;		// 0 solid
	WORD	EMatrl;			// render material 0=plain
	WORD	ELayer;			// layer 0 merge 1 standard
	float	EWidth;			// entity edge width (geometric)
	WORD	EWPlane;		// 3d workplane
	DWORD	ETag;			// entity tag #
}CSTUFF;

// 	===============================================================
// 	2d POINT
// 	===============================================================

#define ET_POINT2		1
#define EC_POINT2		EIC_Point
#define EH_POINT2		EIC_Std

typedef struct
{
	CSTUFF  CStuff;
	GPNT2	Geo;
	char	MType;		// marker type
}POINT2;

// 	===============================================================
// 	2d LINE
// 	===============================================================

#define ET_LINE2		2
#define EC_LINE2		EIC_Linear
#define EH_LINE2		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GLINE2	Geo;
}LINE2;

// 	===============================================================
// 	2d PATH
// 	===============================================================

#define ET_PATH2		3
#define EC_PATH2		EIC_Linear
#define EH_PATH2		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GPATH2	Geo;		
}PATH2;


// 	===============================================================
// 	2d POLY
// 	===============================================================

#define ET_POLY2		4
#define EC_POLY2		EIC_Linear+EIC_Area
#define EH_POLY2		EIC_Std

typedef struct
{
	CSTUFF CStuff;
	GPATH2 Geo;
}POLY2;


// 	A polygon duplicates its first node as the last node,
// 	as this greatly simplifies & speeds up code, for the
// 	minimal loss of memory. This duplicated node IS counted
// 	in nNodes.

typedef struct
{
	CSTUFF CStuff;

	WORD	nNodes;		 // = 5
	GPNT2	P1;		
	GPNT2	P2;		
	GPNT2	P3;		
	GPNT2	P4;		
	GPNT2	P5;		
}BOX2;

// 	===============================================================
// 	2d TEXT
// 	===============================================================

#define ET_TEXT2		5
#define EC_TEXT2		EIC_Point
#define EH_TEXT2		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GTEXT2	Geo;
}TEXT2;

// 	===============================================================
// 	2d CIRCLE
// 	===============================================================

#define ET_CIR2			6
#define EC_CIR2			EIC_Linear+EIC_Area+EIC_Curve
#define EH_CIR2			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF CStuff;
	GCIR2		Geo;
}CIR2;


// 	===============================================================
// 	2d ARC
// 	===============================================================

#define ET_ARC2			7
#define EC_ARC2			EIC_Linear+EIC_Curve
#define EH_ARC2			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF	CStuff;
	GARC2		Geo;
}ARC2;


// 	===============================================================
// 	2d ELLIPSE
// 	===============================================================

#define ET_ELP2			8
#define EC_ELP2			EIC_Linear+EIC_Area+EIC_Curve
#define EH_ELP2			EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GELP2		Geo;	
}ELP2;


// 	===============================================================
// 	2d ELLIPTICAL ARC
// 	===============================================================

#define ET_ELA2		9
#define EC_ELA2			EIC_Linear+EIC_Curve
#define EH_ELA2			EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GELA2		Geo;
}ELA2;


// 	===============================================================
// 	2d GROUP (PART - Group is MASM keyWORD)
// 	===============================================================

#define ET_PART			10
#define EC_PART			EIC_Point+EIC_Multi
#define EH_PART			EIC_Std

typedef struct
{
	CSTUFF CStuff;
}PART;


// 	The entities in the part are in the Part entity's sublist

// 	===============================================================
// 	SYMDEF (2d/3d)
// 	===============================================================

#define ET_SYMDEF		11
#define EC_SYMDEF		EIC_Point+EIC_Multi
#define EH_SYMDEF		EIC_Std+EIC_NoDraw

typedef struct
{
	CSTUFF CStuff;

	DWORD	Flags;			// reserved
	GPNT3	OrgPt;			// origin point of SYMDEF
	char	SName[1];
}SYMDEF;

// 	The entities in the SYMDEF are in the SYMDEF entity's sublist

// 	The SName field is variable length and ends with a 0 byte EOS
// 	marker. SymDef.ERLen is set to just include the EOS byte
// 	Entities are *not* yet translated by -OrgPt.

// 	===============================================================
// 	SYMREF
// 	===============================================================

#define ET_SYMREF		12
#define EC_SYMREF		EIC_Point+EIC_Multi
#define EH_SYMREF		EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	SYMDEF	*DefAdr;
	SymTMat	TM;
	char	 SName[1];
}SYMREF;


// 	The SName field is variable length and ends with a 0 byte EOS
// 	marker. SymRef.ERLen is set to just include the EOS byte

//	DefAdr = 0 will cause the symbol reference to perform a new
//	search for the symbol definition by name.

// 	===============================================================
// 	2d MPOLY
// 	===============================================================

#define ET_MPOLY		13
#define EC_MPOLY		EIC_Multi+EIC_Area
#define EH_MPOLY		EIC_Std

typedef struct
{
	CSTUFF CStuff;
}MPOLY;

// 	The entities in the MPoly are in the MPoly entity's sublist

// 	===============================================================
// 	3d LINE
// 	===============================================================

#define ET_LINE3		14
#define EC_LINE3		EIC_Linear+EIC_3d
#define EH_LINE3		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GLINE3	Geo;
}LINE3;


// 	===============================================================
// 	3d PATH
// 	===============================================================

#define ET_PATH3		15
#define EC_PATH3		EIC_Linear+EIC_3d
#define EH_PATH3		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GPATH3	Geo;		
}PATH3;


// 	===============================================================
// 	3d POLY
// 	===============================================================

#define ET_POLY3		16
#define EC_POLY3		EIC_Linear+EIC_Area+EIC_3d
#define EH_POLY3		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GPATH3	Geo;		
}POLY3;

// 	A polygon duplicates its first node as the last node,
// 	as this greatly simplifies & speeds up code, for the
// 	minimal loss of memory. This duplicated node IS counted
// 	in nNodes.

typedef struct
{
	CSTUFF CStuff;

	WORD	nNodes;		 // = 5
	GPNT3	P1;		
	GPNT3	P2;		
	GPNT3	P3;		
	GPNT3	P4;		
	GPNT3	P5;		
}BOX3;

typedef struct
{
	CSTUFF CStuff;

	WORD	nNodes;		 // = 4
	GPNT3	P1;		
	GPNT3	P2;		
	GPNT3	P3;		
	GPNT3	P4;		
}TRI3;

// 	===============================================================
// 	Panel3 - a 4-sided 3d polygon
// 	===============================================================

typedef struct
{
	CSTUFF CStuff;

	WORD	nNodes;		// total # nodes  = 5
	GPNT3	P1;		
	GPNT3	P2;		
	GPNT3	P3;		
	GPNT3	P4;		
	GPNT3	P5;		
}PANEL3;


// 	===============================================================
// 	3d SOLID
// 	===============================================================

#define ET_SOLID3		17
#define EC_SOLID3		EIC_Volume+EIC_3d
#define EH_SOLID3		EIC_Std

typedef struct
{
	CSTUFF CStuff;
}SOLID3;


// 	The SOLID3 header is followed by one or more sections,
// 	each section being a GPATH3 geometry, with closing node
// 	duplicating the first, and finally a GPATH3 with 0 nodes
// 	indicating the end of the solid.

// 	Note that the solid sections are included in the SOLID3.ERLen
// 	total record size.

// 	===============================================================
// 	3d Surface of Revolution
// 	===============================================================

#define ET_SREV3		18
#define EC_SRev3		EIC_Volume+EIC_3d+EIC_Area+EIC_Multi
#define EH_SRev3		EIC_Std+EIC_Area

typedef struct
{
	CSTUFF CStuff;

	char	ARes;			// angular resolution
	char	LRes;			// linear resolution

	GPNT3	POrg;		// axis origin point
	GPNT3	DVec;		// axis direction vector
	double	AngW;		// revolved arc width 0..2pi
}SREV3;


// 	The entities defining the revolved surface are stored
// 	in the SREV3 entity's sublist. Note there will be no gaps
// 	between multiple entities - they will be connected from
// 	the last node of one to the first node of the next.

// 	===============================================================
// 	3d Tabulated Profile
// 	===============================================================

#define ET_TPRO3		19
#define EC_TPRO3		EIC_Area+EIC_3d+EIC_Multi
#define EH_TPRO3		EIC_Std

typedef struct
{
	GPNT3 Pt;			// tab path 3d point
	double	ryz;		// intersect plane tilt angle
	double	rzx;
	double	sfac;		// section scale factor
}TP3S;

typedef struct
{
	CSTUFF CStuff;

	char	ARes;				// angular resolution
	char	LRes;				// linear resolution

	char	TFlags;			// 1=first end closed 2=last end closed
	                  // 4=parallell extrusion (no tilts)
	WORD	nSecs;			// # sections that follow

	TP3S  Tp3s[1];    // 	nSecs copies of the TP3S section here
}TPRO3;


// 	The entities in the sublist define the tabulated cross section.

// 	===============================================================
// 	3d Ruled Surface
// 	===============================================================

#define ET_RSURF3		20
#define EC_RSURF3		EIC_Area+EIC_3d+EIC_Multi
#define EH_RSURF3		EIC_Std

typedef struct
{
	CSTUFF CStuff;

	char	ARes;		// angular resolution
	char	LRes;		// linear resolution
	char	Flags;	// 1=flip ent1 scan 1..0
			 					// 2=flip ent2 scan 1..0
}RSURF3;


// 	The entities in the RSURF entity's sublist define the
// 	edges of the ruled surface.

// 	===============================================================
// 	3d CIRCLE
// 	===============================================================

#define ET_CIR3			21
#define EC_CIR3			EIC_Linear+EIC_Area+EIC_3d+EIC_Curve
#define EH_CIR3			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF	CStuff;
	GCIR3	Geo;		
}CIR3;

// 	===============================================================
// 	3d ARC
// 	===============================================================

#define ET_ARC3			22
#define EC_ARC3			EIC_Linear+EIC_3d+EIC_Curve
#define EH_ARC3			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF	CStuff;
	GARC3	Geo;		
}ARC3;

// 	===============================================================
// 	3d SPHERE
// 	===============================================================

#define ET_SPH3			23
#define EC_SPH3			EIC_Volume+EIC_3d+EIC_Curve
#define EH_SPH3			EIC_Std

typedef struct
{
	CSTUFF CStuff;

	char	LRes;			// longitude resolution
	char	MRes;			// Meridian resolution
	char	SFlags;		// 1=dome has a hole
	char	SType;		// 0=sphere,1=dome,2=dome section

	GSPH3	Geo;		
}SPH3;


// 	===============================================================
// 	3d MPOLY
// 	===============================================================

#define ET_MPOLY3		24
#define EC_MPOLY3		EIC_Area+EIC_Multi+EIC_3d
#define EH_MPOLY3		EIC_Std

typedef struct
{
	CSTUFF CStuff;
}MPOLY3;


// 	The entities in the 3d MPoly are in the MPoly entity's sublist

// 	===============================================================
// 	3d CYLINDER
// 	===============================================================

#define ET_CYL3			25
#define EC_CYL3			EIC_Volume+EIC_3d+EIC_Curve+EIC_Multi
#define EH_CYL3			EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	char		LRes;		// longitude resolution
	char		ARes;		// Angualr resolution
	char		Flags;		// 1=1st endcap, 2=2nd endcap

	GCYL3		Geo;		// geometry
}CYL3;

// 	===============================================================
// 	2d SPLINE
// 	===============================================================

#define ET_SPLINE2		26
#define EC_SPLINE2		EIC_Linear+EIC_Curve
#define EH_SPLINE2		EIC_Std

typedef struct
{
	CSTUFF CStuff;
	GSPLINE2 Geo;
}SPLINE2;

// 	===============================================================
// 	2d SPoly
// 	===============================================================

#define ET_SPOLY2		27
#define EC_SPOLY2		EIC_Linear+EIC_Curve
#define EH_SPOLY2		EIC_Std

typedef struct
{
	CSTUFF CStuff;
	GSPLINE2 Geo;		
}SPOLY2;

//	SPoly2 geometry does NOT duplicate the first node

// 	===============================================================
// 	2d DIML - Linear Dimension
// 	===============================================================

#define ET_DIML2		28
#define EC_DIML2		EIC_Note
#define EH_DIML2		EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	DSTY	DimDS;		// dimension style data
 
	GPNT2	POrg;		// dimension origin point
	GPNT2	PTxt;		// user-fixed text position
	double	BAng;		// baseline bearing angle
	double	LLen0;		// leader #0 length to geo
	double	LLen1;		// leader #1 length to geo
	double	Dist;		// dim distance ldr0->ldr1
	double	TValue;		// forced dim display value if <>0.0
					 	// else display dim=abs(Dist)
	char	ACode;		// arrows 0=center 1=left 2=right
	char	DCode;		// see below
	char	DText;		// ANSIZ dim fixed text
}DIML2;

//	DimL2.DCode flags:

#define LDC_PRL			1	// parallel linear dimension
#define LDC_FIXTP		2	// PTxt is user-fixed text origin

// 	===============================================================
// 	2d DIMR/DIMD - Circular Dimension
// 	===============================================================

#define ET_DIMC2	29
#define EC_DIMC2	EIC_Note
#define EH_DIMC2	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	DSTY	DimDS;			// dimension style data

	GPNT2	PCen;			// conic center point
	GPNT2	PArw;			// dimension arrow point
	GPNT2	PBend;			// dimension bend point
	GPNT2	PLdr;			// dimension leader endpoint
	GPNT2	PTxt;			// user-fixed text position
	double	TValue;			// forced dim display value if <>0.0
				 			// else display dim=abs(Dist)
	TREF	TagRef;			// tag of reference entity (0=none)
	char	DCode;			// see below
	char	DText;			// ANSIZ dim fixed text
}DIMC2;

// 	DIMC2.DCode flags:

#define CDC_DIA		1		// diameter dimension
#define CDC_RAD		2		// radius dimension
#define CDC_FIXTP	4		// PTxt is user-fixed text origin
#define CDC_PBEND	8		// PBend point is valid
#define CDC_PLDR	16		// PLdr point is valid

// 	===============================================================
// 	2d DIMA - Angular Dimension
// 	===============================================================

#define ET_DIMA2	30
#define EC_DIMA2	EIC_Note
#define EH_DIMA2	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	DSTY	DimDS;			// dimension style data

	GPNT2	PCen;			// arc center point
	double	Radius;			// arc radius
	double	SAng;			// arc start angle
	double	AngW;			// arc width
	double	L0Rad;			// radius to inside start leader pt
	double	L1Rad;			// radius to inside end leader pt
 
	GPNT2	PTxt;			// user-fixed text position
	double	TValue;			// forced dim display value if <>0.0
							//	else display dim=AngW
	char	ACode;			// arrows 0=by DSty 1=left 2=right
	char	DCode;			// see below
	char	DText;			// ANSIZ dim fixed text
}DIMA2;

// 	DIMA2.DCode flags:

#define ADC_LDR0	1		// start leader exists
#define ADC_LDR1	2		// end leader exists
#define ADC_FIXTP	4		// PTxt is user-fixed text origin


// 	===============================================================
// 	2d DIMO - Ordinate Dimensions
// 	===============================================================

#define ET_DIMO2	31
#define EC_DIMO2	EIC_Note
#define EH_DIMO2	EIC_Std

typedef struct
{
	double	Dist;				// dist +- from origin
	double	LLen;				// leader +- length from baseline
}ODSet;

typedef struct
{
	CSTUFF	CStuff;

	DSTY	DimDS;		// dimension style data

	WORD	nLdrs;		// # of leader lines
	float	BAng;			// baseline angle
	GPNT2	POrg;			// baseline origin
	double	OLen;		// origin +- leader len to geometry
}DIMO2;

// 	===============================================================
// 	XP Custom Entity
// 	===============================================================

#define ET_XP		32
#define EC_XP		0
#define EH_XP		0

typedef struct
{
	CSTUFF	CStuff;

	DWORD	Xpid;		// XP ID # for custom entity SVC
      					// xp-specific data starts here
}XPENT;


// 	===============================================================
// 	Action Entity - Drawing actuated button macro
// 	===============================================================

#define ET_ACT2		33
#define EC_ACT2		0
#define EH_ACT2		0

typedef struct
{
	CSTUFF	CStuff;

	GWIN2	Win;		// action pick window
	GTEXT2	Geo;		// action text
}ACT2;

// 	===============================================================
// 	Arrow Entity
// 	===============================================================

#define ET_ARROW	34
#define EC_ARROW	EIC_Multi
#define EH_ARROW	0

typedef struct
{
	CSTUFF	CStuff;

	ASTY	Sty;		// arrowhead style
	DWORD	flags;		// flags
}ARROW;

// 	Arrow.flags values:

#define ARROW_T0	1		// draw arrow at T0 end
#define ARROW_T1	2		// draw arrow at T1 end

// 	The arrow entity has a sublist, and the first entity
// 	within it has the arrowheads attached at either end.

// 	===============================================================
// 	XREF- External File Reference Instance
// 	===============================================================

#define ET_XREF		35
#define EC_XREF		0
#define EH_XREF		0

typedef struct
{
	CSTUFF	CStuff;

	DWORD	id;			// XREF id (link to XRefIB)
	DWORD	flags;		// instance flags
	DWORD	rsvd1;		// reserved
	DWORD	rsvd2;		// reserved
	TMat	TM;			// insert transform
}XREF;

// 	Values for XREF.flags:

#define XREF_OVL	1	// overlay file
#define XREF_PART	2	// part file


// 	===============================================================
// 	SHEET- Sheet Group Instance
// 	===============================================================

#define ET_SHEET	36
#define EC_SHEET	0
#define EH_SHEET	0

typedef struct
{
	CSTUFF	CStuff;

	DWORD	Flags;		// sheet flags
	char	SName;		// ANSIZ Sheet name
}SHEET;

// 	Values for SHEET.flags:

#define SHT_CUR			1		// 1 = sheet is current
#define SHT_VIS			2		// 2 = sheet is visible
#define SHT_LAYOUT		4		// 4 = sheet is ACAD layout
#define SHT_NEW		0x100		// new added sheet (erase if cancel)
#define SHT_DEL		0x200		// del sheet if ok

// 	===============================================================
// 	2d Atrib2
// 	===============================================================

#define ET_ATRIB2	37
#define EC_ATRIB2	EIC_Point
#define EH_ATRIB2	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	WORD	Flags;			// attribute flags
	DWORD	TagOfst;		// offset of tag text
	DWORD	PrmpOfst;		// offset of prompt text
	DWORD	ValOfst;		// offset of value text

	GTEXT2	Geo;			// Attribute display text geometry
}ATRIB2;

// 	The Atrib2 header structure is followed by 3 ANSIZ text
// 	strings, The first of which is part of the Geo text geometry,
// 	and is the text to be displayed. This is the Tag name text outside
// 	any SymRef, and either the Value or Tag name text, depending on
// 	the AT_DSPTAG bit, if inside a SymRef.

// 	A Symbol Reference will have the prototype attribute
// 	entity in its definition, and a "use this one" copy
// 	in its sublist. Only the one in its sublist will be
// 	drawn.

// 	Values for Atrib2.flags:

#define AT_NOPRMP			 1	// value is constant - no prompting
#define AT_HIDDEN			 2	// attribute not displayed in SYMREF
#define AT_DSPTAG			 4	// always display the tag name text
#define AT_REF			0x8000	// atrib is a SYMREF instance entity
#define AT_NOTRAN			 8	// no SYMREF transform

// 	When a symbol containing an attribute is entered, the prompt
// 	tag, and default value is displayed. The user may change
// 	the tag and value fields as desired.


// 	===============================================================
// 	NOTE: Note entity structure (non-drawing entity)
// 	===============================================================
#define NoteVer		0				// structure version #

#define ET_NOTE		0x81		// entity type code

typedef struct
{
	CSTUFF	CStuff;

	char	Version;	// version #
	char	Flags;		// options flags
 
	char	NName;		// Note Name

	char	NData;		// note data (ANSIZ multiline)
}NOTE;

// 	===============================================================
// 	2d Wall
// 	===============================================================

#define ET_WALL2	38
#define EC_WALL2	EIC_Point
#define EH_WALL2	EIC_Std

typedef struct
{
	WORD		ID;			// node id #
	WORD		nSpans;	// # spans connected to this node
	WORD		nVisit;	// traversal counter
	double	X;			// node location
	double	Y;		
}WALL2N;

typedef struct
{
	WORD	ToID;				// exit to node #
	char	flags;		
	float	Thickness;	// exit thickness
	float	Ang;				// exit angle
	DWORD	XData;			// extra data
}WALL2S;

typedef struct
{
	CSTUFF	CStuff;

	WORD	NextID;		// next node id

	WALL2N WData[1];		// ID=0=end of node list
}WALL2;


// 	Values for Wall2S.flags

#define Wall_Exit	1		// span exit has been taken

#define Wall_CF		0		// span is center from
#define Wall_CT		0		// span is center to
#define Wall_LF		4		// span is left from
#define Wall_LT		8		// span is left to
#define Wall_RF		8		// span is right from
#define Wall_RT		4		// span is right to

// 	Each node record has its Wall2N.nSpans span records immediatly
// 	following it before the next node record. After the last
// 	node reocrd is a word=0 end marker (Wall2N.ID=0).

// 	===============================================================
// 	SmartSym Control Points Entity
// 	===============================================================

#define ET_CTRLP	39
#define EC_CTRLP	EIC_Point
#define EH_CTRLP	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	GPNT3	P1;				// ctrl point 1
	GPNT3	P2;				// ctrl point 2
	GPNT3	P3;				// ctrl point 3
// 
	DWORD	CFlags;		// control type flags
}CTRLP;

// 	Values for CtrlP.CFlags:

#define CP_ALN			 1		// align bearing with p1->p2
#define CP_CUT			 2		// cut at p1->p2
#define CP_SCALEY		 4		// scale Y to fit
#define CP_UDSCALE	 8		// use dyntrak scale
#define CP_OFFSET		16		// offset from place point

// 	===============================================================

// 	===============================================================
// 	Animate Point Of View Entity
// 	===============================================================

#define ET_APOV		40
#define EC_APOV		EIC_Point
#define EH_APOV		EIC_Std

#define APOV_VER	0		// structure version

typedef struct
{
	CSTUFF	CStuff;

	DWORD	ver;				// APOV structure version
	DWORD	flags;			// animation flags
 
	GPNT3	AtPt;				// look at point (orbit)
	GPNT3	UpDV;				// look up direction vector
 
	DWORD	nSteps;			// # of steps per loop
	DWORD	rsvd;				// for expansion
 
	char	AName[1];		// animation path name
}APOV;

// 	The APOV entity has a sublist containing the animation
// 	camera path entity

// 	Values for APOV.flags:

#define AP_LookAt	1		// look at point (orbit)

// 	===============================================================
// 	3d SPLINE
// 	===============================================================

#define ET_SPLINE3	41
#define EC_SPLINE3	EIC_Linear+EIC_Curve
#define EH_SPLINE3	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	GSPLINE3 Geo;		
}SPLINE3;

// 	===============================================================
// 	3d SPOLY
// 	===============================================================

#define ET_SPOLY3	42
#define EC_SPOLY3	EIC_Linear+EIC_Curve
#define EH_SPOLY3	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	GSPLINE3 Geo;		
}SPOLY3;

// 	SPoly3 geometry does NOT duplicate the first node

// 	===============================================================
// 	3d Mesh Surface
// 	===============================================================

#define ET_MSURF	43
#define EC_MSURF	EIC_3d+EIC_Area
#define EH_MSURF	EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GMESH3	Geo;
}MSURF;

// 	===============================================================
// 	3d Coons Surface
// 	===============================================================

#define ET_CSURF3	44
#define EC_CSURF3	EIC_Area+EIC_3d+EIC_Curve
#define EH_CSURF3	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	char	ARes;		// angular resolution
	char	LRes;		// linear resolution
}CSURF3;

// 	The 4 entities in the CSURF entity's sublist define the
// 	4 edges of the coons surface.

// 	===============================================================
// 	2d Bezier Curve
// 	===============================================================

#define ET_BEZ2		45
#define EC_BEZ2		EIC_Linear+EIC_Curve
#define EH_BEZ2		EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	GBEZ2 Geo;		
}BEZ2;

// 	===============================================================
// 	3d Bezier Curve
// 	===============================================================

#define ET_BEZ3		46
#define EC_BEZ3		EIC_Linear+EIC_Curve
#define EH_BEZ3		EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	GBEZ3 Geo;
}BEZ3;

// 	===============================================================
// 	VIEWPORT
// 	===============================================================

#define ET_VPORT	47
#define EC_VPORT	EIC_Area+EIC_Multi+EIC_3d
#define EH_VPORT	EIC_Std

typedef struct
{
	CSTUFF	CStuff;

	DWORD	flags;		// options bits
	VIEWP	View;		// view spec for viewport
	DWORD	ShtTag;		// tag # of sheet (0=common/all)
	DWORD	ShtAdr;		// address of sheet (0=unknown)

	float	VPScale;	// fixed display scale
 
	DWORD	LyrOvr;		// layer overrides
 						// + = show id
 						// - = hide id
 						// 0 = end of list (merge can't hide)
}VPORT;

// 	VP Flag values:

#define VP_Scale	1	// use VPScale for view scale
#define VP_FitH		2	// scale to fit width
#define VP_FitV		4	// scale to fit height
#define VP_Frame	8	// show viewport frame entities

// 	The VPORT entity's sublist defines the viewport boundaries.

// 	===============================================================
// 	3d TEXT
// 	===============================================================

#define ET_TEXT3		48
#define EC_TEXT3		EIC_Point
#define EH_TEXT3		EIC_Std

typedef struct
{
	CSTUFF	CStuff;
	GTEXT3	Geo;
}TEXT3;

// 	===============================================================
// 	2d/3d TEXT TABLE
// 	===============================================================

#define ET_TTABLE		49
#define EC_TTABLE		EIC_Point
#define EH_TTABLE		EIC_Std

// 	===============================================================
// 	3d TEXT ITEM
// 	===============================================================

#define ET_TITEM		50
#define EC_TITEM		EIC_Point
#define EH_TITEM		EIC_Std

// 	===============================================================
// 	2d/3d CARD (experimental)
// 	===============================================================

#define ET_CARD			50
#define EC_CARD			EIC_Point
#define EH_CARD			EIC_Std

// 	===============================================================
// 	3d Ellipse
// 	===============================================================

#define ET_ELP3			52
#define EC_ELP3			EIC_Linear+EIC_Area+EIC_3d+EIC_Curve
#define EH_ELP3			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF	CStuff;
	GELP3	Geo;		
}ELP3;

// 	===============================================================
// 	3d Elliptical Arc
// 	===============================================================

#define ET_ELA3			53
#define EC_ELA3			EIC_Linear+EIC_3d+EIC_Curve
#define EH_ELA3			EIC_Std+EIC_NUTGrow

typedef struct
{
	CSTUFF	CStuff;
	GELA3	Geo;		
}ELA3;

// 	===============================================================
// 	Entrec definition
// 	===============================================================

typedef union 
{
	CSTUFF		CStuff;
	POINT2		Point;
	LINE2			Line;
	PATH2			Path;
	POLY2			Poly;
	TEXT2			Text;
	CIR2			Cir;
	ARC2			Arc;
	ELP2			Elp;
	ELA2			Ela;
	PART			Part;
	SYMDEF		SymDef;
	SYMREF		SymRef;
	MPOLY			MPoly;
	LINE3			Line3;
	PATH3			Path3;
	POLY3			Poly3;
	SOLID3		Solid3;
	SREV3			SRev3;
	TPRO3			TPro3;
	RSURF3		RSurf3;
	CIR3			Cir3;
	ARC3			Arc3;
	SPH3			Sph3;
	MPOLY3		MPoly3;
	CYL3			Cyl3;
	SPLINE2		Spline;
	SPOLY2		SPoly;
	DIML2 		DimL2;
	DIMC2 		DimC2;
	DIMA2 		DimA2;
	DIMO2 		DimO2;
	XPENT     XPEnt;
	ACT2      Act;
	ARROW     Arrow;
	XREF			XRef;
	SHEET     Sheet;
	ATRIB2    Atrib;
	NOTE			Note;
	WALL2     Wall;
	CTRLP     CtrlP;
	APOV      Apov;
	SPLINE3		Spline3;
	SPOLY3    SPoly3;
	MSURF     MSurf;
	CSURF3		CSurf3;
	BEZ2			Bez;
	BEZ3			Bez3;
	VPORT     VPort;
	TEXT3		Text3;
	ELP3		Elip3;
	ELA3		ElA3;
} *pENTREC;

// 	===============================================================

// Macro to allocate space for structures that have dynamic length.
// Examples:
// 
//   A path named pPath with 4 nodes:    ESTRUC(PATH2, pPath, 4*8);
//   A text named pText with 12 chars:   ESTRUC(TEXT2, pText, 12);
//

#define ESTRUC(_Type, _Name, _ExtraBytes)  \
		  char _##_Name##Memory[sizeof(_Type)+_ExtraBytes]; \
			_Type *_Name = (_Type *)_##_Name##Memory


#endif